# Archief Checkbox Selectie Functies - Digitaal Notariaat

## Overzicht

Het Archief-systeem is uitgebreid met uitgebreide checkbox selectie functies die gebruikers in staat stellen meerdere records voor te selecteren voor verschillende acties.

## Nieuwe Functies

### 1. **Individuele Record Selectie**

Elk record in het archief heeft nu een checkbox voor individuele selectie:

```typescript
<input
  type="checkbox"
  checked={selectedRecords.has(record.id)}
  onChange={() => handleSelectRecord(record.id)}
  className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2"
/>
```

**Functies:**
- ✅ Individuele records selecteren/deselecteren
- ✅ Visuele selectie status
- ✅ Integratie met wettelijke bescherming

### 2. **"Alles selecteren" Checkbox**

Een globale checkbox om alle beschikbare records te selecteren:

```typescript
<div className="flex items-center space-x-3 p-3 bg-gray-50 border rounded-lg">
  <input
    type="checkbox"
    checked={selectAllRecords}
    onChange={handleSelectAllRecords}
    className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2"
  />
  <label className="text-sm font-medium text-gray-700">
    Alles selecteren ({records.length} Records)
  </label>
</div>
```

**Functies:**
- ✅ Alle records tegelijk selecteren
- ✅ Alle selecties tegelijk opheffen
- ✅ Dynamische weergave van record aantal

### 3. **Selectie Actie Balk**

Een dynamische actie balk die verschijnt wanneer records zijn geselecteerd:

```typescript
{selectedRecords.size > 0 && (
  <div className="flex items-center justify-between p-3 bg-blue-50 border border-blue-200 rounded-lg">
    <div className="flex items-center space-x-4">
      <span className="text-sm font-medium text-blue-900">
        {selectedRecords.size} Record{selectedRecords.size !== 1 ? 's' : ''} geselecteerd
      </span>
      <button onClick={() => setSelectedRecords(new Set())}>
        Selectie opheffen
      </button>
    </div>
    <div className="flex items-center space-x-2">
      <button onClick={handleExportSelectedRecords}>
        <Download className="w-4 h-4" />
        <span>Exporteren</span>
      </button>
      <button onClick={handleDeleteSelectedRecords}>
        <Trash2 className="w-4 h-4" />
        <span>Verwijderen</span>
      </button>
    </div>
  </div>
)}
```

**Functies:**
- ✅ Weergave van aantal geselecteerde records
- ✅ "Selectie opheffen" knop
- ✅ Export knop voor geselecteerde records
- ✅ Verwijder knop voor geselecteerde records

## Technische Implementatie

### 1. **State Management**

```typescript
// Nieuwe state variabelen
const [selectedRecords, setSelectedRecords] = useState<Set<string>>(new Set());
const [selectAllRecords, setSelectAllRecords] = useState(false);
```

### 2. **Selectie Functies**

```typescript
// Individuele records selecteren
const handleSelectRecord = (recordId: string) => {
  const newSelected = new Set(selectedRecords);
  if (newSelected.has(recordId)) {
    newSelected.delete(recordId);
  } else {
    newSelected.add(recordId);
  }
  setSelectedRecords(newSelected);
};

// Alle records selecteren
const handleSelectAllRecords = () => {
  if (selectAllRecords) {
    setSelectedRecords(new Set());
    setSelectAllRecords(false);
  } else {
    const allRecordIds = records.map(record => record.id);
    setSelectedRecords(new Set(allRecordIds));
    setSelectAllRecords(true);
  }
};
```

### 3. **Actie Functies**

```typescript
// Geselecteerde records verwijderen
const handleDeleteSelectedRecords = () => {
  if (selectedRecords.size === 0) {
    alert('Selecteer ten minste één record.');
    return;
  }

  const selectedArray = Array.from(selectedRecords);
  const protectedRecords = selectedArray.filter(id => 
    secureDB.isRecordLegallyProtected(id)
  );
  
  if (protectedRecords.length > 0) {
    alert(`❌ ${protectedRecords.length} geselecteerde records zijn wettelijk beschermd.`);
    return;
  }

  if (confirm(`Weet u zeker dat u ${selectedRecords.size} geselecteerde records wilt verwijderen?`)) {
    // Verwijder logica
  }
};

// Geselecteerde records exporteren
const handleExportSelectedRecords = () => {
  if (selectedRecords.size === 0) {
    alert('Selecteer ten minste één record.');
    return;
  }

  const selectedArray = Array.from(selectedRecords);
  const selectedData = records.filter(record => 
    selectedArray.includes(record.id)
  );
  
  // Export logica
};
```

## Gebruikersinterface

### 1. **Checkbox Ontwerp**

- **Kleur**: Blauw (#3B82F6) voor geselecteerde checkboxes
- **Hover Effect**: Focus ring bij hover
- **Toegankelijkheid**: Juiste labels en ARIA attributen

### 2. **Visuele Indicatoren**

- **Wettelijk beschermde gegevens**: Rood schild icoon met waarschuwing
- **Selectie status**: Blauwe achtergrondkleur voor geselecteerde rijen
- **Actie balk**: Blauwe achtergrondkleur met duidelijke knoppen

### 3. **Responsive Design**

- **Mobiel**: Checkboxes blijven toegankelijk
- **Desktop**: Optimale rangschikking van alle elementen
- **Tablet**: Aangepaste groottes voor touch interactie

## Beveiligingsfuncties

### 1. **Wettelijke Bescherming**

```typescript
// Controle op wettelijk beschermde gegevens
const protectedRecords = selectedArray.filter(id => 
  secureDB.isRecordLegallyProtected(id)
);

if (protectedRecords.length > 0) {
  alert(`❌ ${protectedRecords.length} geselecteerde records zijn wettelijk beschermd.`);
  return;
}
```

### 2. **Bevestigingsdialogen**

- **Verwijdering**: Bevestiging met aantal te verwijderen records
- **Export**: Bevestiging met aantal te exporteren records
- **Beschermingswaarschuwingen**: Duidelijke meldingen over niet-verwijderbare gegevens

### 3. **Audit Trail**

Alle bulk acties worden gelogd:
```typescript
auditTrail: [
  {
    action: 'Bulk verwijdering uitgevoerd',
    timestamp: new Date(),
    user: 'Gebruiker',
    details: `${selectedRecords.size} records verwijderd`
  }
]
```

## Gebruikersbegeleiding

### 1. **Intuïtieve Bediening**

- **Checkboxes**: Standaard verwacht gedrag
- **"Alles selecteren"**: Duidelijke labeling van functie
- **Actie balk**: Verschijnt alleen bij selectie

### 2. **Feedback Systeem**

- **Visuele feedback**: Geselecteerde rijen worden gemarkeerd
- **Aantal weergave**: Duidelijke indicatie van geselecteerde records
- **Status berichten**: Bevestigingen voor alle acties

### 3. **Foutafhandeling**

- **Lege selectie**: Waarschuwing voor acties zonder selectie
- **Beschermde gegevens**: Duidelijke meldingen over niet-verwijderbare gegevens
- **Bevestiging**: Prompt voor kritieke acties

## Voordelen

### 1. **Efficiëntie**

- ✅ **Bulk operaties**: Meerdere records gelijktijdig verwerken
- ✅ **Tijdsbesparing**: Geen individuele selectie voor veel records
- ✅ **Flexibiliteit**: Individuele of globale selectie mogelijk

### 2. **Beveiliging**

- ✅ **Wettelijke bescherming**: Automatische controle op beschermde gegevens
- ✅ **Bevestigingen**: Prompts voor kritieke acties
- ✅ **Audit trail**: Volledige logging van alle acties

### 3. **Gebruiksvriendelijkheid**

- ✅ **Intuïtieve bediening**: Standaard checkbox functionaliteit
- ✅ **Duidelijke feedback**: Visuele en tekstuele bevestigingen
- ✅ **Responsive design**: Werkt op alle apparaten

## Conclusie

De nieuwe checkbox selectie functies in het Archief bieden:

🎯 **Efficiënte bulk operaties** voor betere workflows
🛡️ **Veilige gegevensbeheer** met wettelijke bescherming
👥 **Gebruiksvriendelijke bediening** met intuïtieve interface
📊 **Volledige controle** over record selectie en acties

Het systeem stelt gebruikers in staat om snel en veilig met meerdere records te werken, terwijl wettelijke compliance en gegevensbeveiliging gewaarborgd blijven. 